package main

import (
	"errors"
	"flag"
	"fmt"
	"math"
	"math/bits"
	"os"
	"strconv"
	"strings"
)

var (
	ErrWrongNumberOfOctets = errors.New("wrong number of octets")
	ErrOctetToLarge        = errors.New("octet to large")
)

func parseOctets(input string) (IPAddr, error) {
	var octets IPAddr
	var splitted = strings.Split(input, ".")
	if len(splitted) != 4 {
		return IPAddr{}, ErrWrongNumberOfOctets
	}
	for i, octetStr := range splitted {
		if octet, err := strconv.Atoi(octetStr); err == nil {
			if octet > math.MaxUint8 {
				return IPAddr{}, ErrOctetToLarge
			}
			octets[i] = byte(octet)
		} else {
			return IPAddr{}, err
		}
	}
	return octets, nil
}

func main() {
	var (
		ipaddr   IPAddr
		netmask  IPAddr
		netaddr  IPAddr
		cidr     bool
		bitcount int
	)

	flag.BoolVar(&cidr, "cidr", false, "output in cidr notation")
	flag.Usage = func() {
		fmt.Fprintf(flag.CommandLine.Output(), "Usage: %s [-cidr] <ipv4addr> <netmask>\n", os.Args[0])
		flag.PrintDefaults()
	}
	flag.Parse()

	if flag.NArg() != 2 {
		flag.Usage()
		os.Exit(1)
	}

	/*
		for i, value := range os.Args {
			fmt.Printf("%d: %s\n", i, value)
		}
	*/

	if ip, err := parseOctets(flag.Arg(0)); err == nil {
		ipaddr = ip
	} else {
		fmt.Fprint(os.Stderr, err)
		if errors.Is(err, ErrWrongNumberOfOctets) {
			os.Exit(6)
		} else if errors.Is(err, ErrOctetToLarge) {
			os.Exit(7)
		} else {
			os.Exit(5)
		}
	}

	if nm, err := parseOctets(flag.Arg(1)); err == nil {
		netmask = nm
	} else {
		fmt.Fprint(os.Stderr, err)
		os.Exit(7)
	}

	for i := range ipaddr {
		netaddr[i] = ipaddr[i] & netmask[i]
	}

	bitcount = bits.OnesCount8(netmask[0]) +
		bits.OnesCount8(netmask[1]) +
		bits.OnesCount8(netmask[2]) +
		bits.OnesCount8(netmask[3])

	fmt.Printf("IP address: %s\n", ipaddr)
	fmt.Printf("netmask: %s\n", netmask)
	fmt.Printf("net address: %s", netaddr)
	if cidr {
		fmt.Printf("/%d\n", bitcount)
	} else {
		fmt.Println()
	}
}
